<?php

namespace Modules\Pos\Http\Controllers\Api\V1;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Modules\Core\Http\Controllers\Controller;
use Modules\Order\Services\Order\OrderServiceInterface;
use Modules\Pos\Services\Pos\PosServiceInterface;
use Modules\Support\ApiResponse;
use Modules\User\Services\Customer\CustomerServiceInterface;
use Modules\User\Transformers\Api\V1\SearchCustomerResource;
use Throwable;

class PosController extends Controller
{
    /**
     * Create a new instance of PosController
     *
     * @param PosServiceInterface $service
     * @param OrderServiceInterface $orderService
     * @param CustomerServiceInterface $customerService
     */
    public function __construct(protected PosServiceInterface      $service,
                                protected OrderServiceInterface    $orderService,
                                protected CustomerServiceInterface $customerService
    )
    {
    }

    /**
     * This method retrieves and returns a list of Pos models.
     *
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        return ApiResponse::success($this->service->get());
    }

    /**
     * Get kitchen Viewer data
     *
     * @return JsonResponse
     */
    public function kitchenViewer(): JsonResponse
    {
        return ApiResponse::success($this->service->kitchenViewer());
    }

    /**
     * Search customers
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function searchCustomers(Request $request): JsonResponse
    {
        return ApiResponse::success(
            SearchCustomerResource::collection(
                $this->customerService->search($request->get('query'))
            )
        );
    }

    /**
     * Kitchen move to next status
     *
     * @param int|string $orderId
     * @return JsonResponse
     * @throws Throwable
     */
    public function kitchenMoveToNextStatus(int|string $orderId): JsonResponse
    {
        $newStatus = $this->orderService->kitchenMoveToNextStatus($orderId);

        return ApiResponse::success(message: __("order::messages.order_update_status_to_successfully", ['status' => $newStatus->trans()]));
    }
}
