import type { AxiosError } from 'axios'
import type { Cart, CartResponse } from '@/contracts/types/cart.ts'
import { ref } from 'vue'
import { useI18n } from 'vue-i18n'
import { useToast } from 'vue-toastification'
import * as editOrderCartApi from '@/api/cart/editOrderCart.ts'
import * as cartApi from '@/api/cart/posCart.ts'

const initCart: Cart = {
  response: {
    items: [],
    quantity: 0,
    subTotal: { amount: 0, formatted: '0 JOD', currency: 'JOD' },
    orderType: {
      id: null,
      name: null,
    },
    taxes: [],
    total: { amount: 0, formatted: '0 JOD', currency: 'JOD' },
    customer: null,
  },
  paymentMethods: [],
  payments: {},
  notes: null,
  guestCount: 1,
  paymentType: 'full',
  amountToBePaid: 0,
  carPlate: null,
  refund_payment_method: null,
  carDescription: null,
  scheduledAt: null,
  posRegisterId: null,
  tableId: null,
  mode: 'create',
  orderId: null,
  tableInfo: null,
  checkoutStep: 'cart',
  searchQuery: null,
  disabled: false,
}

export const usePosCart = () => {
  const toast = useToast()
  const { t } = useI18n()
  const cart = ref<Cart>(structuredClone(initCart))

  const initEditOrder = (data: Record<string, any>) => {
    cart.value.mode = 'edit'
    cart.value.response = data.cart
    cart.value.orderId = data.order.id
    cart.value.posRegisterId = data.order.pos_register_id
    cart.value.guestCount = data.order.guest_count
    cart.value.notes = data.order.notes
    cart.value.scheduledAt = data.order.details?.scheduled_at
    cart.value.carPlate = data.order.details?.car_plate
    cart.value.carDescription = data.order.details?.car_description
    cart.value.tableId = data.order.table.id
  }

  const showError = (error: any) => {
    toast.error((error as AxiosError<{
      message?: string
    }>).response?.data?.message || t('core::errors.an_unexpected_error_occurred'))
  }

  const callApi = async (
    callback: (...args: any[]) => Promise<any>,
    ...params: any[]
  ): Promise<any> => {
    cart.value.disabled = true
    let isSuccess = false
    try {
      cart.value.response = (await callback(...params)).data.body
      isSuccess = true
    } catch (error) {
      showError(error)
    } finally {
      cart.value.disabled = false
    }
    return isSuccess
  }

  const getCart = async () => {
    await callApi(cartApi.index)
  }

  const clear = async () => {
    cart.value.disabled = true
    try {
      resetCart((await cartApi.clear()).data.body)
    } catch (error) {
      showError(error)
    } finally {
      cart.value.disabled = false
    }
  }

  const storeOrderType = async (type: string) => {
    await callApi(cartApi.storeOrderType, type)
  }

  const removeOrderType = async () => {
    await callApi(cartApi.removeOrderType)
  }

  const deleteItem = async (id: string) => {
    await callApi((cart.value.mode === 'edit' ? editOrderCartApi : cartApi).deleteItem, id)
  }

  const updateItem = async (id: string, qty: number) => {
    await callApi((cart.value.mode === 'edit' ? editOrderCartApi : cartApi).updateItem, id, qty)
  }

  const resetCart = (response?: CartResponse) => {
    const newCart = structuredClone(initCart)
    cart.value = {
      ...newCart,
      posRegisterId: cart.value.posRegisterId,
      response: response || newCart.response,
    }
  }

  const storeProduct = (params: Record<string, any>) => {
    return (cart.value.mode === 'edit' ? editOrderCartApi : cartApi).storeItem(params)
  }

  const storeAction = async (id: string, action: string, qty: number) => {
    await callApi(editOrderCartApi.storeAction, id, action, qty)
  }
  const removeAction = async (id: string, action: string) => {
    await callApi(editOrderCartApi.removeAction, id, action)
  }

  const applyDiscount = async (id: number) => {
    await callApi((cart.value.mode === 'edit' ? editOrderCartApi : cartApi).applyDiscount, id)
  }
  const applyVoucher = async (code: string) => {
    await callApi((cart.value.mode === 'edit' ? editOrderCartApi : cartApi).applyVoucher, code)
  }

  const removeDiscount = async () => {
    await callApi((cart.value.mode === 'edit' ? editOrderCartApi : cartApi).removeDiscount)
  }

  const addCustomer = async (id: number) => {
    await callApi((cart.value.mode === 'edit' ? editOrderCartApi : cartApi).addCustomer, id)
  }

  const applyGift = async (id: number): Promise<boolean> => {
    return await callApi((cart.value.mode === 'edit' ? editOrderCartApi : cartApi).applyGift, id)
  }

  const removeCustomer = async () => {
    await callApi((cart.value.mode === 'edit' ? editOrderCartApi : cartApi).removeCustomer)
  }

  return {
    cart,
    applyDiscount,
    applyGift,
    removeDiscount,
    addCustomer,
    removeCustomer,
    getCart,
    updateItem,
    storeItem: storeProduct,
    deleteItem,
    clear,
    storeOrderType,
    removeOrderType,
    resetCart,
    storeAction,
    removeAction,
    initEditOrder,
    applyVoucher,
  }
}
