<?php

namespace Modules\Order\Http\Requests\Api\V1;

use Illuminate\Validation\Rule;
use Modules\Cart\CartItem;
use Modules\Cart\Facades\EditOrderCart;
use Modules\Core\Http\Requests\Request;
use Modules\Menu\Models\Menu;
use Modules\Order\Services\Order\OrderService;
use Modules\Payment\Enums\RefundPaymentMethod;

class UpdateOrderRequest extends Request
{
    /**
     * Get the validation rules that apply to the request.
     */
    public function rules(): array
    {
        $order = app(OrderService::class)->findOrFail($this->route('orderId'));

        $branchId = $order->branch_id;

        $menu = Menu::getActiveMenu($branchId);

        abort_if(is_null($menu), 400, __("pos::messages.menu_is_not_active"));

        return [
            "products" => "required|array",
            "products.*" => "bail|required|numeric|exists:products,id,deleted_at,NULL,is_active,1,menu_id,$menu->id",
            "notes" => "nullable|string|max:1000",
            "guest_count" => "nullable|numeric|min:1",
            "refund_payment_method" => ["nullable", Rule::in(RefundPaymentMethod::values())],
            "car_plate" => "nullable|string|max:200",
            "car_description" => "nullable|string|max:200",
            "pos_register_id" => "bail|required|numeric|exists:pos_registers,id,deleted_at,NULL,is_active,1,branch_id,$order->branch_id",
            "scheduled_at" => "required_if:type,pre_order|nullable|date|after_or_equal:today|date_format:Y-m-d H:i:s"];
    }

    /** @inheritDoc */
    public function validationData(): array
    {
        return [
            ...parent::validationData(),
            "products" => EditOrderCart::items()->map(fn(CartItem $cartItem) => $cartItem->product->id)->toArray(),
        ];
    }

    /** @inheritDoc */
    protected function availableAttributes(): string
    {
        return "order::attributes.orders";
    }
}
