<?php

namespace Modules\Printer\Services\Printer;

use App\Forkiva;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Modules\Branch\Models\Branch;
use Modules\Category\Models\Category;
use Modules\Menu\Models\Menu;
use Modules\Pos\Models\PosRegister;
use Modules\Printer\Drivers\TcpPrinterDriver;
use Modules\Printer\Enum\PrinterConnectionType;
use Modules\Printer\Enum\PrinterRole;
use Modules\Printer\Models\Printer;
use Modules\Printer\Templates\KitchenBufferTemplate;
use Modules\Printer\Templates\ReceiptBufferTemplate;
use Modules\Support\GlobalStructureFilters;

class PrinterService implements PrinterServiceInterface
{
    /** @inheritDoc */
    public function label(): string
    {
        return __("printer::printers.printer");
    }

    /** @inheritDoc */
    public function show(int $id): Printer
    {
        return $this->findOrFail($id);
    }

    /** @inheritDoc */
    public function findOrFail(int $id): Builder|array|EloquentCollection|Printer
    {
        return $this->getModel()
            ->query()
            ->withoutGlobalActive()
            ->findOrFail($id);
    }

    /** @inheritDoc */
    public function getModel(): Printer
    {
        return new ($this->model());
    }

    /** @inheritDoc */
    public function model(): string
    {
        return Printer::class;
    }

    /** @inheritDoc */
    public function store(array $data): Printer
    {
        return $this->getModel()->query()->create($data);
    }

    /** @inheritDoc */
    public function update(int $id, array $data): Printer
    {
        $printer = $this->findOrFail($id);
        $printer->update($data);

        return $printer;
    }

    /** @inheritDoc */
    public function destroy(int|array|string $ids): bool
    {
        return $this->getModel()
            ->query()
            ->withoutGlobalActive()
            ->whereIn("id", parseIds($ids))
            ->delete() ?: false;
    }

    /** @inheritDoc */
    public function getStructureFilters(): array
    {
        $branchFilter = GlobalStructureFilters::branch();
        return [
            ...(is_null($branchFilter) ? [] : [$branchFilter]),
            [
                "key" => 'role',
                "label" => __('printer::printers.filters.role'),
                "type" => 'select',
                "options" => PrinterRole::toArrayTrans(),
            ],
            [
                "key" => 'connection_type',
                "label" => __('printer::printers.filters.connection_type'),
                "type" => 'select',
                "options" => PrinterConnectionType::toArrayTrans(),
            ],
            GlobalStructureFilters::active(),
            GlobalStructureFilters::from(),
            GlobalStructureFilters::to(),
        ];
    }

    /** @inheritDoc */
    public function getFormMeta(?int $branchId = null): array
    {
        if (is_null($branchId)) {
            return [
                "branches" => Branch::list(),
                "roles" => PrinterRole::toArrayTrans(),
                "connection_types" => PrinterConnectionType::toArrayTrans(),
            ];
        } else {
            $menu = Menu::getActiveMenu($branchId);

            return [
                "categories" => !is_null($menu) ? Category::treeListWithSlug($menu->id) : [],
                "pos_registers" => !is_null($menu) ? PosRegister::list($menu->branch_id) : [],
            ];
        }
    }

    /** @inheritDoc */
    public function dispatch(PrinterRole $role, array $data, ?PrinterConnectionType $connectionType = null): void
    {
        $printers = $this->for($role, $data['branch_id'], $connectionType);

        $buffer = $role === PrinterRole::Cashier
            ? ReceiptBufferTemplate::build($data)
            : null;

        /** @var Printer $printer */
        foreach ($printers as $printer) {
            if ($role == PrinterRole::Kitchen) {
                $items = $data['items'];
                if (!empty($printer->category_slugs)) {
                    $items = array_filter(
                        $items,
                        function ($item) use ($printer) {
                            return count(array_intersect($item['category_slugs'], $printer->category_slugs)) > 0;
                        }
                    );
                }
                if (empty($items)) {
                    continue;
                }
                $buffer = KitchenBufferTemplate::build([...$data, "items" => $items]);
            } elseif (
                $role == PrinterRole::Cashier
                && !is_null($printer->pos_register_id)
                && $printer->pos_register_id != $data['pos_register_id']
            ) {
                continue;
            }

            if ($printer->connection_type === PrinterConnectionType::Tcp) {
                (new TcpPrinterDriver($printer->options['host'], $printer->options['port']))->send($buffer);
            }
        }
    }

    /** @inheritDoc */
    public function for(PrinterRole $role, int $branchId, ?PrinterConnectionType $connectionType = null): Collection
    {
        return $this->getModel()
            ->query()
            ->withOutGlobalBranchPermission()
            ->where('branch_id', $branchId)
            ->where('role', $role)
            ->when(
                !is_null($connectionType),
                fn($query) => $query->where('connection_type', $connectionType)
            )
            ->get();
    }

    /** @inheritDoc */
    public function get(array $filters = [], ?array $sorts = []): LengthAwarePaginator
    {
        return $this->getModel()
            ->query()
            ->with(["branch:id,name"])
            ->withoutGlobalActive()
            ->filters($filters)
            ->sortBy($sorts)
            ->paginate(Forkiva::paginate())
            ->withQueryString();
    }
}
